/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/

package cnrg.itx.gtwy;
import cnrg.itx.datax.*;
import cnrg.itx.datax.devices.*;
import java.net.*;

/**
 * Takes care of sending voice data from the primary channel from the phone to the computer network.
 * 
 * @author James Wann
 * @version 1.0a
 * @see java.lang.Runnable
 */
public class RecordChannel implements Runnable
{
	static
	{
		System.loadLibrary("Dialogic");
	}
	
	/**
	 * The Line instance corresponding to RecordChannel's line.
	 */
	private Line myLine;
	
	/**
	 * The Connection instance needed for data transfer.
	 */
	private Connection myConnection;
	
	/**
	 * The Gateway instance.
	 */
	private Gateway myGate;
	
	/**
	 * @param lin the <code>Line</code> instance corresponding to <code>RecordChannel's</code> line.
	 * Contains information regarding the line.
	 * @param c the <code>Connection</code> instance needed for data transfer with the computer network.
	 * @param g the <code>Gateway</code> instance.  Needed to access line state.
	 * @see Line
	 * @see Gateway
	 * @see cnrg.itx.datax.Connection
	 */
	protected RecordChannel(Line lin, Connection c, Gateway g)
	{
		myLine = lin;	
		myConnection = c;
		myGate = g;
	}
	
	public void run()
	{
		String s = new String();							// the string returned by recordPacket
		boolean isDTMF = true;								// if termination condition is a DTMF digit
		int myLineNumber = myLine.getLineNumber();
		int myChDev = myLine.getChDev();

		Channel myChannel = myConnection.getOutputChannel();
		RecordSource mySource = (RecordSource)myChannel.getSource();
	
		while (isDTMF == true)
		{
			s = recordPacket(mySource, myChDev, myLineNumber);
			
			// Hangup if connection is being terminated
			if (s.equals("HANGUP"))
			{
				try
				{
					myGate.hangup(myLine);
				}
				catch (GatewayException e)
				{
					System.out.println("<RecordChannel: " + myLine.getLineNumber() + "> -> " + e.getMessage());
				}
				myGate.tearDownPlayConnection(myLine);
				isDTMF = false;
			}
				
			// Otherwise, DTMF digits are being sent.  Send them.
			else
			{
				try
				{
					myGate.sendDTMF(s, myLine);
				}
				catch (GatewayException e)
				{
					System.out.println("<RecordChannel: " + myLine.getLineNumber() + "> -> " + e.getMessage());
				}
			}
		}
	}
	
	/**
	 * Sends voice data from phone to computer.
	 * 
	 * @param dataChannel the object whose methods are used to send data to a PC.
	 * @param chDev the channel's device handler.
	 * @param lineNumber the channel's corresponding line number.
	 * @return any DTMF digits that were pressed during recording or an indication of a hangup.
	 */
	private native String recordPacket(RecordSource dataChannel, int chDev, int lineNumber);
	
}